/*
 * Copyright (c) 1999-2002,2004,2011,2014 Apple Inc. All Rights Reserved.
 * 
 * @APPLE_LICENSE_HEADER_START@
 * 
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this
 * file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 *
 * cssmerr.h -- Error Code Definitions for CSSM
 */

#ifndef _CSSMERR_H_
#define _CSSMERR_H_  1

#ifdef __cplusplus
extern "C" {
#endif


/*
 * NOTE: To translate CSSM error codes into something vaguely
 * human-readable, use the cssmPerror function in cssmapple.h.
 * This function will also decode other Security layer errors
 * (i.e. those with names like kSec...).
 */


/* Common error codes. */
enum {
	CSSM_BASE_ERROR =				-0x7FFF0000 /* 0x80010000 */
};

enum {
	CSSM_ERRORCODE_MODULE_EXTENT =	0x00000800,
	CSSM_ERRORCODE_CUSTOM_OFFSET =	0x00000400,
	CSSM_ERRORCODE_COMMON_EXTENT =	0x100
};

/* Macros for convertible error code manipulation. */
#define CSSM_ERRCODE(CODE) \
	(((CODE) - CSSM_BASE_ERROR) & (CSSM_ERRORCODE_MODULE_EXTENT - 1)) 

#define CSSM_ERRBASE(CODE) \
	((((CODE) - CSSM_BASE_ERROR) & ~(CSSM_ERRORCODE_MODULE_EXTENT - 1)) + CSSM_BASE_ERROR) 

#define CSSM_ERR_IS_CONVERTIBLE(CODE) \
	(CSSM_ERRCODE(CODE) < CSSM_ERRORCODE_COMMON_EXTENT)

#define CSSM_ERR_TAG(CODE, BASE) \
	(CSSM_ERRCODE(CODE) + (BASE))

/* Error Bases for different module types. */
enum {
	CSSM_CSSM_BASE_ERROR =				CSSM_BASE_ERROR,
	CSSM_CSSM_PRIVATE_ERROR =			CSSM_BASE_ERROR + CSSM_ERRORCODE_CUSTOM_OFFSET,
	CSSM_CSP_BASE_ERROR =				CSSM_CSSM_BASE_ERROR + CSSM_ERRORCODE_MODULE_EXTENT,
	CSSM_CSP_PRIVATE_ERROR =			CSSM_CSP_BASE_ERROR + CSSM_ERRORCODE_CUSTOM_OFFSET,
	CSSM_DL_BASE_ERROR =				CSSM_CSP_BASE_ERROR + CSSM_ERRORCODE_MODULE_EXTENT,
	CSSM_DL_PRIVATE_ERROR =				CSSM_DL_BASE_ERROR + CSSM_ERRORCODE_CUSTOM_OFFSET,
	CSSM_CL_BASE_ERROR =				CSSM_DL_BASE_ERROR + CSSM_ERRORCODE_MODULE_EXTENT,
	CSSM_CL_PRIVATE_ERROR =				CSSM_CL_BASE_ERROR + CSSM_ERRORCODE_CUSTOM_OFFSET,
	CSSM_TP_BASE_ERROR =				CSSM_CL_BASE_ERROR + CSSM_ERRORCODE_MODULE_EXTENT,
	CSSM_TP_PRIVATE_ERROR =				CSSM_TP_BASE_ERROR + CSSM_ERRORCODE_CUSTOM_OFFSET ,
	CSSM_KR_BASE_ERROR =				CSSM_TP_BASE_ERROR + CSSM_ERRORCODE_MODULE_EXTENT,
	CSSM_KR_PRIVATE_ERROR =				CSSM_KR_BASE_ERROR + CSSM_ERRORCODE_CUSTOM_OFFSET,
	CSSM_AC_BASE_ERROR =				CSSM_KR_BASE_ERROR + CSSM_ERRORCODE_MODULE_EXTENT,
	CSSM_AC_PRIVATE_ERROR =				CSSM_AC_BASE_ERROR + CSSM_ERRORCODE_CUSTOM_OFFSET
};

/* XXX @@@ MDS Error Bases same as DL for now. */
enum {
	CSSM_MDS_BASE_ERROR =				CSSM_CSP_BASE_ERROR + CSSM_ERRORCODE_MODULE_EXTENT,
	CSSM_MDS_PRIVATE_ERROR =			CSSM_MDS_BASE_ERROR + CSSM_ERRORCODE_CUSTOM_OFFSET
};

/* General Error Values. */
enum {
	CSSMERR_CSSM_INVALID_ADDIN_HANDLE =     -2147417855,
	CSSMERR_CSSM_NOT_INITIALIZED =          -2147417854,
	CSSMERR_CSSM_INVALID_HANDLE_USAGE =     -2147417853,
	CSSMERR_CSSM_PVC_REFERENT_NOT_FOUND =   -2147417852,
	CSSMERR_CSSM_FUNCTION_INTEGRITY_FAIL =  -2147417851,
};

/* Common Error Codes For All Module Types. */
enum {
	CSSM_ERRCODE_INTERNAL_ERROR =					0x0001,
	CSSM_ERRCODE_MEMORY_ERROR =						0x0002,
	CSSM_ERRCODE_MDS_ERROR =						0x0003,
	CSSM_ERRCODE_INVALID_POINTER =					0x0004,
	CSSM_ERRCODE_INVALID_INPUT_POINTER =			0x0005,
	CSSM_ERRCODE_INVALID_OUTPUT_POINTER =			0x0006,
	CSSM_ERRCODE_FUNCTION_NOT_IMPLEMENTED =			0x0007,
	CSSM_ERRCODE_SELF_CHECK_FAILED =				0x0008,
	CSSM_ERRCODE_OS_ACCESS_DENIED =					0x0009,
	CSSM_ERRCODE_FUNCTION_FAILED =					0x000A,
	CSSM_ERRCODE_MODULE_MANIFEST_VERIFY_FAILED =	0x000B,
	CSSM_ERRCODE_INVALID_GUID =						0x000C
};

/* Common Error Codes for ACLs */
enum {
	CSSM_ERRCODE_OPERATION_AUTH_DENIED =			0x0020,
	CSSM_ERRCODE_OBJECT_USE_AUTH_DENIED =			0x0021,
	CSSM_ERRCODE_OBJECT_MANIP_AUTH_DENIED =			0x0022,
	CSSM_ERRCODE_OBJECT_ACL_NOT_SUPPORTED =			0x0023,
	CSSM_ERRCODE_OBJECT_ACL_REQUIRED =				0x0024,
	CSSM_ERRCODE_INVALID_ACCESS_CREDENTIALS =		0x0025,
	CSSM_ERRCODE_INVALID_ACL_BASE_CERTS =			0x0026,
	CSSM_ERRCODE_ACL_BASE_CERTS_NOT_SUPPORTED =		0x0027,
	CSSM_ERRCODE_INVALID_SAMPLE_VALUE =				0x0028,
	CSSM_ERRCODE_SAMPLE_VALUE_NOT_SUPPORTED =		0x0029,
	CSSM_ERRCODE_INVALID_ACL_SUBJECT_VALUE =		0x002A,
	CSSM_ERRCODE_ACL_SUBJECT_TYPE_NOT_SUPPORTED =	0x002B,
	CSSM_ERRCODE_INVALID_ACL_CHALLENGE_CALLBACK =	0x002C,
	CSSM_ERRCODE_ACL_CHALLENGE_CALLBACK_FAILED =	0x002D,
	CSSM_ERRCODE_INVALID_ACL_ENTRY_TAG =			0x002E,
	CSSM_ERRCODE_ACL_ENTRY_TAG_NOT_FOUND =			0x002F,
	CSSM_ERRCODE_INVALID_ACL_EDIT_MODE =			0x0030,
	CSSM_ERRCODE_ACL_CHANGE_FAILED =				0x0031,
	CSSM_ERRCODE_INVALID_NEW_ACL_ENTRY =			0x0032,
	CSSM_ERRCODE_INVALID_NEW_ACL_OWNER =			0x0033,
	CSSM_ERRCODE_ACL_DELETE_FAILED =				0x0034,
	CSSM_ERRCODE_ACL_REPLACE_FAILED =				0x0035,
	CSSM_ERRCODE_ACL_ADD_FAILED =					0x0036
};

/* Common Error Codes for Specific Data Types */
enum {
	CSSM_ERRCODE_INVALID_CONTEXT_HANDLE =			0x0040,
	CSSM_ERRCODE_INCOMPATIBLE_VERSION =				0x0041,
	CSSM_ERRCODE_INVALID_CERTGROUP_POINTER =		0x0042,
	CSSM_ERRCODE_INVALID_CERT_POINTER =				0x0043,
	CSSM_ERRCODE_INVALID_CRL_POINTER =				0x0044,
	CSSM_ERRCODE_INVALID_FIELD_POINTER =			0x0045,
	CSSM_ERRCODE_INVALID_DATA =						0x0046,
	CSSM_ERRCODE_CRL_ALREADY_SIGNED =				0x0047,
	CSSM_ERRCODE_INVALID_NUMBER_OF_FIELDS =			0x0048,
	CSSM_ERRCODE_VERIFICATION_FAILURE =				0x0049,
	CSSM_ERRCODE_INVALID_DB_HANDLE =				0x004A,
	CSSM_ERRCODE_PRIVILEGE_NOT_GRANTED =			0x004B,
	CSSM_ERRCODE_INVALID_DB_LIST =					0x004C,
	CSSM_ERRCODE_INVALID_DB_LIST_POINTER =			0x004D,
	CSSM_ERRCODE_UNKNOWN_FORMAT =					0x004E,
	CSSM_ERRCODE_UNKNOWN_TAG =						0x004F,
	CSSM_ERRCODE_INVALID_CSP_HANDLE =				0x0050,
	CSSM_ERRCODE_INVALID_DL_HANDLE =				0x0051,
	CSSM_ERRCODE_INVALID_CL_HANDLE =				0x0052,
	CSSM_ERRCODE_INVALID_TP_HANDLE =				0x0053,
	CSSM_ERRCODE_INVALID_KR_HANDLE =				0x0054,
	CSSM_ERRCODE_INVALID_AC_HANDLE =				0x0055,
	CSSM_ERRCODE_INVALID_PASSTHROUGH_ID =			0x0056,
	CSSM_ERRCODE_INVALID_NETWORK_ADDR =				0x0057,
	CSSM_ERRCODE_INVALID_CRYPTO_DATA =				0x0058
};

/* CSSM Error Values Derived from Common Error Codes For All Module Types. */
enum {
	CSSMERR_CSSM_INTERNAL_ERROR =                   -2147418111,
	CSSMERR_CSSM_MEMORY_ERROR =                     -2147418110,
	CSSMERR_CSSM_MDS_ERROR =                        -2147418109,
	CSSMERR_CSSM_INVALID_POINTER =                  -2147418108,
	CSSMERR_CSSM_INVALID_INPUT_POINTER =            -2147418107,
	CSSMERR_CSSM_INVALID_OUTPUT_POINTER =           -2147418106,
	CSSMERR_CSSM_FUNCTION_NOT_IMPLEMENTED =         -2147418105,
	CSSMERR_CSSM_SELF_CHECK_FAILED =                -2147418104,
	CSSMERR_CSSM_OS_ACCESS_DENIED =                 -2147418103,
	CSSMERR_CSSM_FUNCTION_FAILED =                  -2147418102,
	CSSMERR_CSSM_MODULE_MANIFEST_VERIFY_FAILED =    -2147418101,
	CSSMERR_CSSM_INVALID_GUID =                     -2147418100,
};

/* CSSM Error Values for Specific Data Types. */
enum {
	CSSMERR_CSSM_INVALID_CONTEXT_HANDLE =           -2147418048,
	CSSMERR_CSSM_INCOMPATIBLE_VERSION =             -2147418047,
	CSSMERR_CSSM_PRIVILEGE_NOT_GRANTED =            -2147418037,
};

/* CSSM Module-Specific Error Values */
enum {
	CSSM_CSSM_BASE_CSSM_ERROR =
		CSSM_CSSM_BASE_ERROR + CSSM_ERRORCODE_COMMON_EXTENT + 0x10,
	CSSMERR_CSSM_SCOPE_NOT_SUPPORTED =				-2147417839,
	CSSMERR_CSSM_PVC_ALREADY_CONFIGURED =			-2147417838,
	CSSMERR_CSSM_INVALID_PVC =						-2147417837,
	CSSMERR_CSSM_EMM_LOAD_FAILED =					-2147417836,
	CSSMERR_CSSM_EMM_UNLOAD_FAILED =				-2147417835,
	CSSMERR_CSSM_ADDIN_LOAD_FAILED =				-2147417834,
	CSSMERR_CSSM_INVALID_KEY_HIERARCHY =			-2147417833,
	CSSMERR_CSSM_ADDIN_UNLOAD_FAILED =				-2147417832,
	CSSMERR_CSSM_LIB_REF_NOT_FOUND =				-2147417831,
	CSSMERR_CSSM_INVALID_ADDIN_FUNCTION_TABLE =		-2147417830,
	CSSMERR_CSSM_EMM_AUTHENTICATE_FAILED =			-2147417829,
	CSSMERR_CSSM_ADDIN_AUTHENTICATE_FAILED =		-2147417828,
	CSSMERR_CSSM_INVALID_SERVICE_MASK =				-2147417827,
	CSSMERR_CSSM_MODULE_NOT_LOADED =				-2147417826,
	CSSMERR_CSSM_INVALID_SUBSERVICEID =				-2147417825,
	CSSMERR_CSSM_BUFFER_TOO_SMALL =					-2147417824,
	CSSMERR_CSSM_INVALID_ATTRIBUTE =				-2147417823,
	CSSMERR_CSSM_ATTRIBUTE_NOT_IN_CONTEXT =			-2147417822,
	CSSMERR_CSSM_MODULE_MANAGER_INITIALIZE_FAIL =	-2147417821,
	CSSMERR_CSSM_MODULE_MANAGER_NOT_FOUND =			-2147417820,
	CSSMERR_CSSM_EVENT_NOTIFICATION_CALLBACK_NOT_FOUND = -2147417819,
};

/* CSP Error Values Derived from Common Error Codes For All Module Types. */
enum {
	CSSMERR_CSP_INTERNAL_ERROR =                    -2147416063,
	CSSMERR_CSP_MEMORY_ERROR =                      -2147416062,
	CSSMERR_CSP_MDS_ERROR =                         -2147416061,
    CSSMERR_CSP_INVALID_POINTER =                   -2147416060,
	CSSMERR_CSP_INVALID_INPUT_POINTER =             -2147416059,
	CSSMERR_CSP_INVALID_OUTPUT_POINTER =            -2147416058,
	CSSMERR_CSP_FUNCTION_NOT_IMPLEMENTED =          -2147416057,
	CSSMERR_CSP_SELF_CHECK_FAILED =                 -2147416056,
	CSSMERR_CSP_OS_ACCESS_DENIED =                  -2147416055,
	CSSMERR_CSP_FUNCTION_FAILED =                   -2147416054,
};

/* CSP Error Values Derived from ACL-based Error Codes. */
enum {
	CSSMERR_CSP_OPERATION_AUTH_DENIED =             -2147416032,
	CSSMERR_CSP_OBJECT_USE_AUTH_DENIED =            -2147416031,
	CSSMERR_CSP_OBJECT_MANIP_AUTH_DENIED =          -2147416030,
	CSSMERR_CSP_OBJECT_ACL_NOT_SUPPORTED =          -2147416029,
	CSSMERR_CSP_OBJECT_ACL_REQUIRED =               -2147416028,
	CSSMERR_CSP_INVALID_ACCESS_CREDENTIALS =        -2147416027,
	CSSMERR_CSP_INVALID_ACL_BASE_CERTS =            -2147416026,
	CSSMERR_CSP_ACL_BASE_CERTS_NOT_SUPPORTED =      -2147416025,
	CSSMERR_CSP_INVALID_SAMPLE_VALUE =              -2147416024,
	CSSMERR_CSP_SAMPLE_VALUE_NOT_SUPPORTED =        -2147416023,
	CSSMERR_CSP_INVALID_ACL_SUBJECT_VALUE =         -2147416022,
	CSSMERR_CSP_ACL_SUBJECT_TYPE_NOT_SUPPORTED =    -2147416021,
	CSSMERR_CSP_INVALID_ACL_CHALLENGE_CALLBACK =    -2147416020,
	CSSMERR_CSP_ACL_CHALLENGE_CALLBACK_FAILED =     -2147416019,
	CSSMERR_CSP_INVALID_ACL_ENTRY_TAG =             -2147416018,
	CSSMERR_CSP_ACL_ENTRY_TAG_NOT_FOUND =           -2147416017,
	CSSMERR_CSP_INVALID_ACL_EDIT_MODE =             -2147416016,
	CSSMERR_CSP_ACL_CHANGE_FAILED =                 -2147416015,
	CSSMERR_CSP_INVALID_NEW_ACL_ENTRY =             -2147416014,
	CSSMERR_CSP_INVALID_NEW_ACL_OWNER =             -2147416013,
	CSSMERR_CSP_ACL_DELETE_FAILED =                 -2147416012,
	CSSMERR_CSP_ACL_REPLACE_FAILED =                -2147416011,
	CSSMERR_CSP_ACL_ADD_FAILED =                    -2147416010,
};

/* CSP Error Values for Specific Data Types. */
enum {
	CSSMERR_CSP_INVALID_CONTEXT_HANDLE =            -2147416000,
	CSSMERR_CSP_PRIVILEGE_NOT_GRANTED =             -2147415989,
	CSSMERR_CSP_INVALID_DATA =                      -2147415994,
	CSSMERR_CSP_INVALID_PASSTHROUGH_ID =            -2147415978,
	CSSMERR_CSP_INVALID_CRYPTO_DATA =               -2147415976,
};

/* CSP Module-Specific Error Values */
enum {
	/* General CSP Error Values */
	CSSM_CSP_BASE_CSP_ERROR =
		CSSM_CSP_BASE_ERROR + CSSM_ERRORCODE_COMMON_EXTENT,
	CSSMERR_CSP_INPUT_LENGTH_ERROR =				-2147415807,
	CSSMERR_CSP_OUTPUT_LENGTH_ERROR =				-2147415806,
	CSSMERR_CSP_PRIVILEGE_NOT_SUPPORTED =			-2147415805,
	CSSMERR_CSP_DEVICE_ERROR =						-2147415804,
	CSSMERR_CSP_DEVICE_MEMORY_ERROR =				-2147415803,
	CSSMERR_CSP_ATTACH_HANDLE_BUSY =				-2147415802,
	CSSMERR_CSP_NOT_LOGGED_IN =						-2147415801,
	CSSMERR_CSP_INVALID_KEY =						-2147415792,
	CSSMERR_CSP_INVALID_KEY_REFERENCE =				-2147415791,
	CSSMERR_CSP_INVALID_KEY_CLASS =					-2147415790,
	CSSMERR_CSP_ALGID_MISMATCH =					-2147415789,
	CSSMERR_CSP_KEY_USAGE_INCORRECT =				-2147415788,
	CSSMERR_CSP_KEY_BLOB_TYPE_INCORRECT =			-2147415787,
	CSSMERR_CSP_KEY_HEADER_INCONSISTENT =			-2147415786,
	CSSMERR_CSP_UNSUPPORTED_KEY_FORMAT =			-2147415785,
	CSSMERR_CSP_UNSUPPORTED_KEY_SIZE =				-2147415784,
	CSSMERR_CSP_INVALID_KEY_POINTER =				-2147415783,
	CSSMERR_CSP_INVALID_KEYUSAGE_MASK =				-2147415782,
	CSSMERR_CSP_UNSUPPORTED_KEYUSAGE_MASK =			-2147415781,
	CSSMERR_CSP_INVALID_KEYATTR_MASK =				-2147415780,
	CSSMERR_CSP_UNSUPPORTED_KEYATTR_MASK =			-2147415779,
	CSSMERR_CSP_INVALID_KEY_LABEL =					-2147415778,
	CSSMERR_CSP_UNSUPPORTED_KEY_LABEL =				-2147415777,
	CSSMERR_CSP_INVALID_KEY_FORMAT =				-2147415776,

	CSSMERR_CSP_INVALID_DATA_COUNT =				-2147415768,
	CSSMERR_CSP_VECTOR_OF_BUFS_UNSUPPORTED =		-2147415767,
	CSSMERR_CSP_INVALID_INPUT_VECTOR =				-2147415766,
	CSSMERR_CSP_INVALID_OUTPUT_VECTOR =				-2147415765,

	CSSMERR_CSP_INVALID_CONTEXT =					-2147415760,
	CSSMERR_CSP_INVALID_ALGORITHM =					-2147415759,
	CSSMERR_CSP_INVALID_ATTR_KEY = 					-2147415754,
	CSSMERR_CSP_MISSING_ATTR_KEY = 					-2147415753,
	CSSMERR_CSP_INVALID_ATTR_INIT_VECTOR = 			-2147415752,
	CSSMERR_CSP_MISSING_ATTR_INIT_VECTOR = 			-2147415751,
	CSSMERR_CSP_INVALID_ATTR_SALT = 				-2147415750,
	CSSMERR_CSP_MISSING_ATTR_SALT = 				-2147415749,
	CSSMERR_CSP_INVALID_ATTR_PADDING = 				-2147415748,
	CSSMERR_CSP_MISSING_ATTR_PADDING = 				-2147415747,
	CSSMERR_CSP_INVALID_ATTR_RANDOM = 				-2147415746,
	CSSMERR_CSP_MISSING_ATTR_RANDOM = 				-2147415745,
	CSSMERR_CSP_INVALID_ATTR_SEED = 				-2147415744,
	CSSMERR_CSP_MISSING_ATTR_SEED = 				-2147415743,
	CSSMERR_CSP_INVALID_ATTR_PASSPHRASE = 			-2147415742,
	CSSMERR_CSP_MISSING_ATTR_PASSPHRASE = 			-2147415741,
	CSSMERR_CSP_INVALID_ATTR_KEY_LENGTH = 			-2147415740,
	CSSMERR_CSP_MISSING_ATTR_KEY_LENGTH = 			-2147415739,
	CSSMERR_CSP_INVALID_ATTR_BLOCK_SIZE = 			-2147415738,
	CSSMERR_CSP_MISSING_ATTR_BLOCK_SIZE = 			-2147415737,
	CSSMERR_CSP_INVALID_ATTR_OUTPUT_SIZE = 			-2147415708,
	CSSMERR_CSP_MISSING_ATTR_OUTPUT_SIZE = 			-2147415707,
	CSSMERR_CSP_INVALID_ATTR_ROUNDS = 				-2147415706,
	CSSMERR_CSP_MISSING_ATTR_ROUNDS = 				-2147415705,
	CSSMERR_CSP_INVALID_ATTR_ALG_PARAMS = 			-2147415704,
	CSSMERR_CSP_MISSING_ATTR_ALG_PARAMS = 			-2147415703,
	CSSMERR_CSP_INVALID_ATTR_LABEL = 				-2147415702,
	CSSMERR_CSP_MISSING_ATTR_LABEL = 				-2147415701,
	CSSMERR_CSP_INVALID_ATTR_KEY_TYPE = 			-2147415700,
	CSSMERR_CSP_MISSING_ATTR_KEY_TYPE = 			-2147415699,
	CSSMERR_CSP_INVALID_ATTR_MODE = 				-2147415698,
	CSSMERR_CSP_MISSING_ATTR_MODE = 				-2147415697,
	CSSMERR_CSP_INVALID_ATTR_EFFECTIVE_BITS = 		-2147415696,
	CSSMERR_CSP_MISSING_ATTR_EFFECTIVE_BITS = 		-2147415695,
	CSSMERR_CSP_INVALID_ATTR_START_DATE = 			-2147415694,
	CSSMERR_CSP_MISSING_ATTR_START_DATE = 			-2147415693,
	CSSMERR_CSP_INVALID_ATTR_END_DATE = 			-2147415692,
	CSSMERR_CSP_MISSING_ATTR_END_DATE = 			-2147415691,
	CSSMERR_CSP_INVALID_ATTR_VERSION = 				-2147415690,
	CSSMERR_CSP_MISSING_ATTR_VERSION = 				-2147415689,
	CSSMERR_CSP_INVALID_ATTR_PRIME = 				-2147415688,
	CSSMERR_CSP_MISSING_ATTR_PRIME = 				-2147415687,
	CSSMERR_CSP_INVALID_ATTR_BASE = 				-2147415686,
	CSSMERR_CSP_MISSING_ATTR_BASE = 				-2147415685,
	CSSMERR_CSP_INVALID_ATTR_SUBPRIME = 			-2147415684,
	CSSMERR_CSP_MISSING_ATTR_SUBPRIME = 			-2147415683,
	CSSMERR_CSP_INVALID_ATTR_ITERATION_COUNT = 		-2147415682,
	CSSMERR_CSP_MISSING_ATTR_ITERATION_COUNT = 		-2147415681,
	CSSMERR_CSP_INVALID_ATTR_DL_DB_HANDLE = 		-2147415680,
	CSSMERR_CSP_MISSING_ATTR_DL_DB_HANDLE = 		-2147415679,
	CSSMERR_CSP_INVALID_ATTR_ACCESS_CREDENTIALS = 	-2147415678,
	CSSMERR_CSP_MISSING_ATTR_ACCESS_CREDENTIALS = 	-2147415677,
	CSSMERR_CSP_INVALID_ATTR_PUBLIC_KEY_FORMAT = 	-2147415676,
	CSSMERR_CSP_MISSING_ATTR_PUBLIC_KEY_FORMAT = 	-2147415675,
	CSSMERR_CSP_INVALID_ATTR_PRIVATE_KEY_FORMAT = 	-2147415674,
	CSSMERR_CSP_MISSING_ATTR_PRIVATE_KEY_FORMAT = 	-2147415673,
	CSSMERR_CSP_INVALID_ATTR_SYMMETRIC_KEY_FORMAT = -2147415672,
	CSSMERR_CSP_MISSING_ATTR_SYMMETRIC_KEY_FORMAT = -2147415671,
	CSSMERR_CSP_INVALID_ATTR_WRAPPED_KEY_FORMAT = 	-2147415670,
	CSSMERR_CSP_MISSING_ATTR_WRAPPED_KEY_FORMAT = 	-2147415669,

	CSSMERR_CSP_STAGED_OPERATION_IN_PROGRESS =		-2147415736,
	CSSMERR_CSP_STAGED_OPERATION_NOT_STARTED =		-2147415735,
	CSSMERR_CSP_VERIFY_FAILED =						-2147415734,
	CSSMERR_CSP_INVALID_SIGNATURE =					-2147415733,
	CSSMERR_CSP_QUERY_SIZE_UNKNOWN =				-2147415732,
	CSSMERR_CSP_BLOCK_SIZE_MISMATCH =				-2147415731,
	CSSMERR_CSP_PRIVATE_KEY_NOT_FOUND =				-2147415730,
	CSSMERR_CSP_PUBLIC_KEY_INCONSISTENT =			-2147415729,
	CSSMERR_CSP_DEVICE_VERIFY_FAILED =				-2147415728,
	CSSMERR_CSP_INVALID_LOGIN_NAME =				-2147415727,
	CSSMERR_CSP_ALREADY_LOGGED_IN =					-2147415726,
	CSSMERR_CSP_PRIVATE_KEY_ALREADY_EXISTS =		-2147415725,
	CSSMERR_CSP_KEY_LABEL_ALREADY_EXISTS =			-2147415724,
	CSSMERR_CSP_INVALID_DIGEST_ALGORITHM =			-2147415723,
	CSSMERR_CSP_CRYPTO_DATA_CALLBACK_FAILED =		-2147415722,
};


/* TP Error Values Derived from Common Error Codes For All Module Types. */
enum {
	CSSMERR_TP_INTERNAL_ERROR =                 -2147409919,
	CSSMERR_TP_MEMORY_ERROR =                   -2147409918,
	CSSMERR_TP_MDS_ERROR =                      -2147409917,
	CSSMERR_TP_INVALID_POINTER =                -2147409916,
	CSSMERR_TP_INVALID_INPUT_POINTER =          -2147409915,
	CSSMERR_TP_INVALID_OUTPUT_POINTER =         -2147409914,
	CSSMERR_TP_FUNCTION_NOT_IMPLEMENTED =       -2147409913,
	CSSMERR_TP_SELF_CHECK_FAILED =              -2147409912,
	CSSMERR_TP_OS_ACCESS_DENIED =               -2147409911,
	CSSMERR_TP_FUNCTION_FAILED =                -2147409910,
	CSSMERR_TP_INVALID_CONTEXT_HANDLE =         -2147409856,
	CSSMERR_TP_INVALID_DATA =                   -2147409850,
	CSSMERR_TP_INVALID_DB_LIST =                -2147409844,
	CSSMERR_TP_INVALID_CERTGROUP_POINTER =      -2147409854,
	CSSMERR_TP_INVALID_CERT_POINTER =           -2147409853,
	CSSMERR_TP_INVALID_CRL_POINTER =            -2147409852,
	CSSMERR_TP_INVALID_FIELD_POINTER =          -2147409851,
	CSSMERR_TP_INVALID_NETWORK_ADDR =           -2147409833,
	CSSMERR_TP_CRL_ALREADY_SIGNED =             -2147409849,
	CSSMERR_TP_INVALID_NUMBER_OF_FIELDS =       -2147409848,
	CSSMERR_TP_VERIFICATION_FAILURE =           -2147409847,
	CSSMERR_TP_INVALID_DB_HANDLE =              -2147409846,
	CSSMERR_TP_UNKNOWN_FORMAT =                 -2147409842,
	CSSMERR_TP_UNKNOWN_TAG =                    -2147409841,
	CSSMERR_TP_INVALID_PASSTHROUGH_ID =         -2147409834,
	CSSMERR_TP_INVALID_CSP_HANDLE =             -2147409840,
	CSSMERR_TP_INVALID_DL_HANDLE =              -2147409839,
	CSSMERR_TP_INVALID_CL_HANDLE =              -2147409838,
	CSSMERR_TP_INVALID_DB_LIST_POINTER =        -2147409843,
};

/* TP Module-Specific Error Values */
enum {
	CSSM_TP_BASE_TP_ERROR =
		CSSM_TP_BASE_ERROR + CSSM_ERRORCODE_COMMON_EXTENT,
	CSSMERR_TP_INVALID_CALLERAUTH_CONTEXT_POINTER =	-2147409663,
	CSSMERR_TP_INVALID_IDENTIFIER_POINTER =			-2147409662,
	CSSMERR_TP_INVALID_KEYCACHE_HANDLE =			-2147409661,
	CSSMERR_TP_INVALID_CERTGROUP =					-2147409660,
	CSSMERR_TP_INVALID_CRLGROUP =					-2147409659,
	CSSMERR_TP_INVALID_CRLGROUP_POINTER =			-2147409658,
	CSSMERR_TP_AUTHENTICATION_FAILED =				-2147409657,
	CSSMERR_TP_CERTGROUP_INCOMPLETE =				-2147409656,
	CSSMERR_TP_CERTIFICATE_CANT_OPERATE =			-2147409655,
	CSSMERR_TP_CERT_EXPIRED =						-2147409654,
	CSSMERR_TP_CERT_NOT_VALID_YET =					-2147409653,
	CSSMERR_TP_CERT_REVOKED =						-2147409652,
	CSSMERR_TP_CERT_SUSPENDED =						-2147409651,
	CSSMERR_TP_INSUFFICIENT_CREDENTIALS =			-2147409650,
	CSSMERR_TP_INVALID_ACTION =						-2147409649,
	CSSMERR_TP_INVALID_ACTION_DATA =				-2147409648,
	CSSMERR_TP_INVALID_ANCHOR_CERT =				-2147409646,
	CSSMERR_TP_INVALID_AUTHORITY =					-2147409645,
	CSSMERR_TP_VERIFY_ACTION_FAILED =				-2147409644,
	CSSMERR_TP_INVALID_CERTIFICATE =				-2147409643,
	CSSMERR_TP_INVALID_CERT_AUTHORITY =				-2147409642,
	CSSMERR_TP_INVALID_CRL_AUTHORITY =				-2147409641,
	CSSMERR_TP_INVALID_CRL_ENCODING =				-2147409640,
	CSSMERR_TP_INVALID_CRL_TYPE =					-2147409639,
	CSSMERR_TP_INVALID_CRL =						-2147409638,
	CSSMERR_TP_INVALID_FORM_TYPE =					-2147409637,
	CSSMERR_TP_INVALID_ID =							-2147409636,
	CSSMERR_TP_INVALID_IDENTIFIER =					-2147409635,
	CSSMERR_TP_INVALID_INDEX =						-2147409634,
	CSSMERR_TP_INVALID_NAME =						-2147409633,
	CSSMERR_TP_INVALID_POLICY_IDENTIFIERS =			-2147409632,
	CSSMERR_TP_INVALID_TIMESTRING =					-2147409631,
	CSSMERR_TP_INVALID_REASON =						-2147409630,
	CSSMERR_TP_INVALID_REQUEST_INPUTS =				-2147409629,
	CSSMERR_TP_INVALID_RESPONSE_VECTOR =			-2147409628,
	CSSMERR_TP_INVALID_SIGNATURE =					-2147409627,
	CSSMERR_TP_INVALID_STOP_ON_POLICY =				-2147409626,
	CSSMERR_TP_INVALID_CALLBACK =					-2147409625,
	CSSMERR_TP_INVALID_TUPLE =						-2147409624,
	CSSMERR_TP_NOT_SIGNER =							-2147409623,
	CSSMERR_TP_NOT_TRUSTED =						-2147409622,
	CSSMERR_TP_NO_DEFAULT_AUTHORITY =				-2147409621,
	CSSMERR_TP_REJECTED_FORM =						-2147409620,
	CSSMERR_TP_REQUEST_LOST =						-2147409619,
	CSSMERR_TP_REQUEST_REJECTED =					-2147409618,
	CSSMERR_TP_UNSUPPORTED_ADDR_TYPE =				-2147409617,
	CSSMERR_TP_UNSUPPORTED_SERVICE =				-2147409616,
	CSSMERR_TP_INVALID_TUPLEGROUP_POINTER =			-2147409615,
	CSSMERR_TP_INVALID_TUPLEGROUP =					-2147409614,
};

/* AC Error Values Derived from Common Error Codes For All Module Types. */
enum {
	CSSMERR_AC_INTERNAL_ERROR =                     -2147405823,
	CSSMERR_AC_MEMORY_ERROR =                       -2147405822,
	CSSMERR_AC_MDS_ERROR =                          -2147405821,
	CSSMERR_AC_INVALID_POINTER =                    -2147405820,
	CSSMERR_AC_INVALID_INPUT_POINTER =              -2147405819,
	CSSMERR_AC_INVALID_OUTPUT_POINTER =             -2147405818,
	CSSMERR_AC_FUNCTION_NOT_IMPLEMENTED =           -2147405817,
	CSSMERR_AC_SELF_CHECK_FAILED =                  -2147405816,
	CSSMERR_AC_OS_ACCESS_DENIED =                   -2147405815,
	CSSMERR_AC_FUNCTION_FAILED =                    -2147405814,
	CSSMERR_AC_INVALID_CONTEXT_HANDLE =             -2147405760,
	CSSMERR_AC_INVALID_DATA =                       -2147405754,
	CSSMERR_AC_INVALID_DB_LIST =                    -2147405748,
	CSSMERR_AC_INVALID_PASSTHROUGH_ID =             -2147405738,
	CSSMERR_AC_INVALID_DL_HANDLE =                  -2147405743,
	CSSMERR_AC_INVALID_CL_HANDLE =                  -2147405742,
	CSSMERR_AC_INVALID_TP_HANDLE =                  -2147405741,
	CSSMERR_AC_INVALID_DB_HANDLE =                  -2147405750,
	CSSMERR_AC_INVALID_DB_LIST_POINTER =            -2147405747,
};

/* AC Module-Specific Error Values */
enum {
	CSSM_AC_BASE_AC_ERROR =
		CSSM_AC_BASE_ERROR + CSSM_ERRORCODE_COMMON_EXTENT,
	CSSMERR_AC_INVALID_BASE_ACLS =					-2147405567,
	CSSMERR_AC_INVALID_TUPLE_CREDENTIALS =			-2147405566,
	CSSMERR_AC_INVALID_ENCODING =					-2147405565,
	CSSMERR_AC_INVALID_VALIDITY_PERIOD =			-2147405564,
	CSSMERR_AC_INVALID_REQUESTOR =					-2147405563,
	CSSMERR_AC_INVALID_REQUEST_DESCRIPTOR =			-2147405562,
};

/* CL Error Values Derived from Common Error Codes For All Module Types. */
enum {
	CSSMERR_CL_INTERNAL_ERROR =                     -2147411967,
	CSSMERR_CL_MEMORY_ERROR =                       -2147411966,
	CSSMERR_CL_MDS_ERROR =                          -2147411965,
	CSSMERR_CL_INVALID_POINTER =                    -2147411964,
	CSSMERR_CL_INVALID_INPUT_POINTER =              -2147411963,
	CSSMERR_CL_INVALID_OUTPUT_POINTER =             -2147411962,
	CSSMERR_CL_FUNCTION_NOT_IMPLEMENTED =           -2147411961,
	CSSMERR_CL_SELF_CHECK_FAILED =                  -2147411960,
	CSSMERR_CL_OS_ACCESS_DENIED =                   -2147411959,
	CSSMERR_CL_FUNCTION_FAILED =                    -2147411958,
	CSSMERR_CL_INVALID_CONTEXT_HANDLE =             -2147411904,
	CSSMERR_CL_INVALID_CERTGROUP_POINTER =          -2147411902,
	CSSMERR_CL_INVALID_CERT_POINTER =               -2147411901,
	CSSMERR_CL_INVALID_CRL_POINTER =                -2147411900,
	CSSMERR_CL_INVALID_FIELD_POINTER =              -2147411899,
	CSSMERR_CL_INVALID_DATA =                       -2147411898,
	CSSMERR_CL_CRL_ALREADY_SIGNED =                 -2147411897,
	CSSMERR_CL_INVALID_NUMBER_OF_FIELDS =           -2147411896,
	CSSMERR_CL_VERIFICATION_FAILURE =               -2147411895,
	CSSMERR_CL_UNKNOWN_FORMAT =                     -2147411890,
	CSSMERR_CL_UNKNOWN_TAG =                        -2147411889,
	CSSMERR_CL_INVALID_PASSTHROUGH_ID =             -2147411882,
};

/* CL Module-Specific Error Values */
enum {
	CSSM_CL_BASE_CL_ERROR =
		CSSM_CL_BASE_ERROR + CSSM_ERRORCODE_COMMON_EXTENT,
	CSSMERR_CL_INVALID_BUNDLE_POINTER =				-2147411711,
	CSSMERR_CL_INVALID_CACHE_HANDLE =				-2147411710,
	CSSMERR_CL_INVALID_RESULTS_HANDLE =				-2147411709,
	CSSMERR_CL_INVALID_BUNDLE_INFO =				-2147411708,
	CSSMERR_CL_INVALID_CRL_INDEX =					-2147411707,
	CSSMERR_CL_INVALID_SCOPE =						-2147411706,
	CSSMERR_CL_NO_FIELD_VALUES =					-2147411705,
	CSSMERR_CL_SCOPE_NOT_SUPPORTED =				-2147411704,
};

/* DL Error Values Derived from Common Error Codes For All Module Types. */
enum {
	CSSMERR_DL_INTERNAL_ERROR =                     -2147414015,
	CSSMERR_DL_MEMORY_ERROR =                       -2147414014,
	CSSMERR_DL_MDS_ERROR =                          -2147414013,
	CSSMERR_DL_INVALID_POINTER =                    -2147414012,
	CSSMERR_DL_INVALID_INPUT_POINTER =              -2147414011,
	CSSMERR_DL_INVALID_OUTPUT_POINTER =             -2147414010,
	CSSMERR_DL_FUNCTION_NOT_IMPLEMENTED =           -2147414009,
	CSSMERR_DL_SELF_CHECK_FAILED =                  -2147414008,
	CSSMERR_DL_OS_ACCESS_DENIED =                   -2147414007,
	CSSMERR_DL_FUNCTION_FAILED =                    -2147414006,
	CSSMERR_DL_INVALID_CSP_HANDLE =                 -2147413936,
	CSSMERR_DL_INVALID_DL_HANDLE =                  -2147413935,
	CSSMERR_DL_INVALID_CL_HANDLE =                  -2147413934,
	CSSMERR_DL_INVALID_DB_LIST_POINTER =            -2147413939,
};

/* DL Error Values Derived from ACL-based Error Codes. */
enum {
	CSSMERR_DL_OPERATION_AUTH_DENIED =              -2147413984,
	CSSMERR_DL_OBJECT_USE_AUTH_DENIED =             -2147413983,
	CSSMERR_DL_OBJECT_MANIP_AUTH_DENIED =           -2147413982,
	CSSMERR_DL_OBJECT_ACL_NOT_SUPPORTED =           -2147413981,
	CSSMERR_DL_OBJECT_ACL_REQUIRED =                -2147413980,
	CSSMERR_DL_INVALID_ACCESS_CREDENTIALS =         -2147413979,
	CSSMERR_DL_INVALID_ACL_BASE_CERTS =             -2147413978,
	CSSMERR_DL_ACL_BASE_CERTS_NOT_SUPPORTED =       -2147413977,
	CSSMERR_DL_INVALID_SAMPLE_VALUE =               -2147413976,
	CSSMERR_DL_SAMPLE_VALUE_NOT_SUPPORTED =         -2147413975,
	CSSMERR_DL_INVALID_ACL_SUBJECT_VALUE =          -2147413974,
	CSSMERR_DL_ACL_SUBJECT_TYPE_NOT_SUPPORTED =     -2147413973,
	CSSMERR_DL_INVALID_ACL_CHALLENGE_CALLBACK =     -2147413972,
	CSSMERR_DL_ACL_CHALLENGE_CALLBACK_FAILED =      -2147413971,
	CSSMERR_DL_INVALID_ACL_ENTRY_TAG =              -2147413970,
	CSSMERR_DL_ACL_ENTRY_TAG_NOT_FOUND =            -2147413969,
	CSSMERR_DL_INVALID_ACL_EDIT_MODE =              -2147413968,
	CSSMERR_DL_ACL_CHANGE_FAILED =                  -2147413967,
	CSSMERR_DL_INVALID_NEW_ACL_ENTRY =              -2147413966,
	CSSMERR_DL_INVALID_NEW_ACL_OWNER =              -2147413965,
	CSSMERR_DL_ACL_DELETE_FAILED =                  -2147413964,
	CSSMERR_DL_ACL_REPLACE_FAILED =                 -2147413963,
	CSSMERR_DL_ACL_ADD_FAILED =                     -2147413962,
};

/* DL Error Values for Specific Data Types. */
enum {
	CSSMERR_DL_INVALID_DB_HANDLE =                  -2147413942,
	CSSMERR_DL_INVALID_PASSTHROUGH_ID =             -2147413930,
	CSSMERR_DL_INVALID_NETWORK_ADDR =               -2147413929,
};

/* DL Module-Specific Error Values */
enum {
	CSSM_DL_BASE_DL_ERROR =
		CSSM_DL_BASE_ERROR + CSSM_ERRORCODE_COMMON_EXTENT,
	CSSMERR_DL_DATABASE_CORRUPT =					-2147413759,
	CSSMERR_DL_INVALID_RECORD_INDEX =				-2147413752,
	CSSMERR_DL_INVALID_RECORDTYPE =					-2147413751,
	CSSMERR_DL_INVALID_FIELD_NAME =					-2147413750,
	CSSMERR_DL_UNSUPPORTED_FIELD_FORMAT =			-2147413749,
	CSSMERR_DL_UNSUPPORTED_INDEX_INFO =				-2147413748,
	CSSMERR_DL_UNSUPPORTED_LOCALITY =				-2147413747,
	CSSMERR_DL_UNSUPPORTED_NUM_ATTRIBUTES =			-2147413746,
	CSSMERR_DL_UNSUPPORTED_NUM_INDEXES =			-2147413745,
	CSSMERR_DL_UNSUPPORTED_NUM_RECORDTYPES =		-2147413744,
	CSSMERR_DL_UNSUPPORTED_RECORDTYPE =				-2147413743,
	CSSMERR_DL_FIELD_SPECIFIED_MULTIPLE =			-2147413742,
	CSSMERR_DL_INCOMPATIBLE_FIELD_FORMAT =			-2147413741,
	CSSMERR_DL_INVALID_PARSING_MODULE =				-2147413740,
	CSSMERR_DL_INVALID_DB_NAME =					-2147413738,
	CSSMERR_DL_DATASTORE_DOESNOT_EXIST =			-2147413737,
	CSSMERR_DL_DATASTORE_ALREADY_EXISTS =			-2147413736,
	CSSMERR_DL_DB_LOCKED =							-2147413735,
	CSSMERR_DL_DATASTORE_IS_OPEN =					-2147413734,
	CSSMERR_DL_RECORD_NOT_FOUND =					-2147413733,
	CSSMERR_DL_MISSING_VALUE =						-2147413732,
	CSSMERR_DL_UNSUPPORTED_QUERY =					-2147413731,
	CSSMERR_DL_UNSUPPORTED_QUERY_LIMITS =			-2147413730,
	CSSMERR_DL_UNSUPPORTED_NUM_SELECTION_PREDS =	-2147413729,
	CSSMERR_DL_UNSUPPORTED_OPERATOR =				-2147413727,
	CSSMERR_DL_INVALID_RESULTS_HANDLE =				-2147413726,
	CSSMERR_DL_INVALID_DB_LOCATION =				-2147413725,
	CSSMERR_DL_INVALID_ACCESS_REQUEST =				-2147413724,
	CSSMERR_DL_INVALID_INDEX_INFO =					-2147413723,
	CSSMERR_DL_INVALID_SELECTION_TAG =				-2147413722,
	CSSMERR_DL_INVALID_NEW_OWNER =					-2147413721,
	CSSMERR_DL_INVALID_RECORD_UID =					-2147413720,
	CSSMERR_DL_INVALID_UNIQUE_INDEX_DATA =			-2147413719,
	CSSMERR_DL_INVALID_MODIFY_MODE =				-2147413718,
	CSSMERR_DL_INVALID_OPEN_PARAMETERS =			-2147413717,
	CSSMERR_DL_RECORD_MODIFIED =					-2147413716,
	CSSMERR_DL_ENDOFDATA =							-2147413715,
	CSSMERR_DL_INVALID_QUERY =						-2147413714,
	CSSMERR_DL_INVALID_VALUE =						-2147413713,
	CSSMERR_DL_MULTIPLE_VALUES_UNSUPPORTED =		-2147413712,
	CSSMERR_DL_STALE_UNIQUE_RECORD =				-2147413711,
};


#ifdef __cplusplus
}
#endif

#endif /* _CSSMERR_H_ */
