/*
    PSFeed.h
    PubSub

    Copyright 2006, Apple Computer, Inc.  All rights reserved.
*/


#import <Foundation/NSObject.h>
@class NSArray;
@class NSEnumerator;
@class NSString;
@class NSData;
@class NSDate;
@class NSTimeZone;
@class NSError;
@class NSURL;
@class NSXMLElement;


#if MAC_OS_X_VERSION_10_5 <= MAC_OS_X_VERSION_MAX_ALLOWED

@class PSFeedSettings, PSClient;


extern NSString* const PSErrorDomain NS_DEPRECATED(10_5, 10_9, NA, NA);
enum {
    PSInternalError NS_ENUM_DEPRECATED(10_5, 10_9, NA, NA) = 1,                // An internal error occurred
    PSNotAFeedError NS_ENUM_DEPRECATED(10_5, 10_9, NA, NA)                     // Data is not Atom or RSS
} NS_ENUM_DEPRECATED(10_5, 10_9, NA, NA);


/*!
    @enum       PSFeedFormat
    @abstract   The data format of a feed.
    @discussion The PubSub framework handles two types of feeds: RSS and Atom. These are both standards built with XML.
    @constant   These types will track the latest published specifications for RSS and Atom. Their current versions are as follows:
                PSRSSType RSS 2.0, as specified by http://blogs.law.harvard.edu/tech/rss
                PSAtomType Atom 1.0, as specified by http://www.ietf.org/rfc/rfc4287.txt
*/
typedef enum {
    PSUnknownFormat NS_ENUM_DEPRECATED(10_5, 10_9, NA, NA) = 0,
    PSRSSFormat NS_ENUM_DEPRECATED(10_5, 10_9, NA, NA)     = 1,
    PSAtomFormat NS_ENUM_DEPRECATED(10_5, 10_9, NA, NA)    = 2,
} PSFeedFormat NS_ENUM_DEPRECATED(10_5, 10_9, NA, NA);


/*!
    @class      PSFeed
    @discussion A <code>PSFeed</code> object is used for managing RSS and Atom feeds. Each feed object represents
                a single feed. Feed objects can either be created for subscription or parsing. To subscribe to
                a feed, use <code>initWithURL:</code> and then add the newly returned feed object to a client. To use Publication
                Subscription to simply parse a feed, download the feed with <code>NSURL</code> and then use <code>initWithData:URL:</code>.

                Many of the feed's attributes are defined by the RSS 2.0 and Atom 1.0 specs. Any
                attributes defined by the specifications (or nonstandard extensions) that do not appear
                in this API can be retrieved using the <code>XMLRepresentation</code> method.

                Since the values of the RSS and Atom attributes are derived from data received
                from the server, they may be missing or incorrect. If you have the opportunity, use
                attributes generated by the Publication Subscription framework, as it is more reliable.
                For example, using <code>updatedDate</code> to find entries that have been modified 
                since a certain time would be unreliable. Use <code>receivedDate</code> instead. It 
                is not recommended that you subclass <code>PSFeed</code>.
    @attribute 	Framework /System/Library/Frameworks/PubSub.framework
    @seealso 	//apple_ref/doc/uid/TP40004945 Publication Subscription Programming Guide
*/

NS_DEPRECATED(10_5, 10_9, NA, NA)
@interface PSFeed : NSObject
{
    @protected
    id _internal;
}


/*!
    @method     initWithURL:
    @abstract   Returns a temporary feed object initialized from the specified URL.
    @discussion This method does not cause any network activity. The feed's entries and metadata
                will not be available until the feed is added to a client (via the <code>
                addFeedWithURL:</code> method of <code>PSClient</code>) or its update method is called, and the feed has been
                downloaded and parsed.
 
                Temporary feeds are not associated with any <code>PSClient</code>, and their contents will
                not be stored persistently. If desired, this feed can later be made persistent and
                'subscribed to' by calling addFeed: on a <code>PSClient</code> object.
 
                Temporary feeds are not automatically updated on a schedule. Instead, you must
                call "update" whenever you want to check for updates.
    @param      aURL  The URL of the feed.
    @result     A feed from the specified URL.
*/
- (id) initWithURL: (NSURL*)url;


/*!
    @method     initWithData:URL:
    @abstract   Returns a temporary feed object initialized to the specified XML data located at the specified URL.
    @discussion This method does not cause any network activity. The feed is parsed directly from the
                provided XML data. The URL parameter is required only to resolve relative URLs, and
                for future reference if this object is later subscribed to.
 
                See the discussion under the <code>initWithURL:</code> method for more information about
                temporary feeds.
    @param      data  A data object containing the feed. This object can be obtained by using NSURL
                or <code>NSURLConnection</code> to download the feed in its entirety.
    @param      aURL  The URL of the feed.
    @result     A feed parsed from the specified data.
*/
- (id) initWithData:(NSData *)data URL:(NSURL *)aURL;


/*!
    @method    client
    @abstract  Returns the client that the receiver belongs to, or <code>nil</code> if the receiver has not been
               added to any client.
    @result    The client associated with the receiver.
*/
@property (readonly) PSClient * client;


/*! @property identifier 
    @abstract A string that uniquely identifies the feed. (read-only) 
    @discussion This identifier can be used as a key to map between <code>PSFeed</code> objects and your application's own data model objects.
    This identifier is generated internally by the Publication Subscription framework. You should not make any assumptions about its format or size, as they may change.
    Although Atom defines a mandatory <id> element, it cannot be used for this purpose since the RSS equivalent (<guid>) is optional.  Thus, Publication Subscription uses its own mechanism for uniquely identifying feeds. Clients can manually parse the XMLRepresentation if they need the Atom <id> element or RSS <guid> element. */
@property (readonly, retain) NSString * identifier;


/*! @property title 
    @abstract A string that contains the title of a given feed. (read-only) 
    @discussion The returned title corresponds to the <title> element in both Atom and RSS. */
@property (readonly, retain) NSString * title;


/*! @property subtitle 
    @abstract A string that contains the subtitle (or description) of a given feed. (read-only) 
    @discussion The returned subtitle corresponds to the <subtitle> element in Atom and <description> in RSS. */
@property (readonly, retain) NSString * subtitle;


/*! @property rights 
    @abstract The legal information of a given feed. (read-only) 
    @discussion The returned string corresponds to the <rights> elements in Atom and <copyright> in RSS. */
@property (readonly, retain) NSString * rights;


/*! @property dateUpdated 
    @abstract A date that represents the most recent time the feed was modified. (read-only) 
    @discussion The date is stored in the feed and set by the software that creates it. Therefore, it is associated with the feed's server, not the machine your application is running on. The returned date corresponds to the <updated> element in Atom and the <lastBuildDate> element in RSS. */
@property (readonly, retain) NSDate * dateUpdated;


@property (readonly, retain) NSTimeZone * timeZoneUpdated;


/*! @property URL 
    @abstract A URL to the feed's original location. 
    @discussion  */
@property (readonly, retain) NSURL * URL;


/*! @property alternateURL 
    @abstract A URL to an alternate version of the feed resource. (read-only) 
    @discussion Traditionally, the alternate URL corresponds to the homepage associated with the feed. The URL is parsed from the <link rel="alternate"> element in Atom and the <link> element in RSS. */
@property (readonly, retain) NSURL * alternateURL;


/*! @property iconURL 
    @abstract A URL to the feed's icon. (read-only) 
    @discussion Icons should have an aspect ratio of 1:1 and be suitable for presentation at a small size.  The URL corresponds to the <icon> element in Atom.  There is no RSS equivalent. */
@property (readonly, retain) NSURL * iconURL;


/*! @property logoURL 
    @abstract A URL to a feed's logo. (read-only) 
    @discussion Even though Atom states that logos should have an aspect ratio of 2:1, RSS prefers a default size of 88 x 31 pixels.  The returned URL corresponds to the <logo> element in Atom and the <image> element in RSS. */
@property (readonly, retain) NSURL * logoURL;


/*! @property redirectedURL 
    @abstract A URL to the current location of a given feed. (read-only) 
    @discussion The URL is to the feed's current location following any redirects. */
@property (readonly, retain) NSURL * redirectedURL;


/*! @property localDateUpdated 
    @abstract The date when the feed and/or its entries were updated in the local Publication Subscription database. (read-only) 
    @discussion This is the date/time that changes in the feed were <i>received</i>, whereas the <code>dateUpdated</code> is the time that changes were <i>published</i> by the server. */
@property (readonly, retain) NSDate * localDateUpdated;


/*! @property login 
    @abstract The user name used if the feed requires authentication. 
    @discussion  */
@property (retain) NSString * login;


/*! @property feedFormat 
    @abstract The data format of the feed (RSS or Atom). (read-only) 
    @discussion  */
@property (readonly) int feedFormat;



/*!
    @property   refreshing
    @abstract   A Boolean value that indicates whether the feed is currently being refreshed. (read-only)
    @discussion	If this property returns a value of <code>NO</code>, this may be because the feed has finished refreshing or because the feed
                has not started refreshing. To get more accurate output, register for the <code>PSFeedRefreshingNotification</code> notification
                and then call this method.
*/
@property (readonly, getter=isRefreshing) BOOL refreshing;


/*!
    @method     refresh
    @abstract   Forces a refresh of the feed.
    @discussion If any immediate problem is encountered, this method returns <code>NO</code> and sets the error parameter.
             	However, refreshing is an asynchronous operation, and errors are more likely to show up later.
             	These can be accessed via the feed's <code>latestError</code> property after it sends a notification
             	that it's ended refreshing.

                To check on the status of the refresh, register for the <code>PSFeedRefreshingNotification</code> notification
                and then check the <code>isRefreshing</code> property.
*/
- (BOOL) refresh: (NSError**)error;


/*!
    @property   lastError
    @abstract   The most recent error that occurred during the feed update, or <code>nil</code> if it was successful. (read-only)
    @discussion Can be used to obtain failure status such as "Feed is 404", "Server is not responding", etc.
*/
@property (readonly) NSError * lastError;


/*!
    @property   settings
    @abstract   The current feed settings.
    @discussion The client's settings are used whenever a feed's settings are undefined.

                This property attempts to communicate with the PubSubAgent process and causes the agent 
                to launch if it is not already running.  If PubSubAgent cannot be contacted, an exception 
                will be thrown with the name <code>PSException</code>.  To avoid problems, you should 
                always be prepared to catch an exception thrown by this property.
*/
@property (retain) PSFeedSettings * settings;


/*!
    @property   links
    @abstract   An array of link objects that provide links to other URLs that are declared in this feed. (read-only)
    @discussion Feeds and HTML pages can contain both "link" tags (that define explicit links to
                other URLs) and the purposes of those links. This is often used for
                "autodiscovery", where a web page contains a link to its feed.
*/
@property (readonly) NSArray * links;


/*!
    @method     setPassword:
    @abstract   Sets the password to use for a feed whose server requires authentication.
    @discussion The password is stored in the user's default keychain, associated with this feed's
                URL.
 
                There is intentionally no corresponding <code>getPassword</code> method, because only PubSub
                itself needs to access the password.
 
                This method is optional. If no password is found in the keychain when the feed
                is being updated, Publication Subscription will prompt the user for one, and 
                add it to the keychain. However, your application's UI may want to let the user 
                enter the password at the same time as he enters the user name.
                
                This method attempts to communicate with the PubSubAgent process and causes the agent 
                to launch if it is not already running.  If PubSubAgent cannot be contacted, an exception 
                will be thrown with the name <code>PSException</code>.  To avoid problems, you should 
                always be prepared to catch an exception thrown by this method.
*/
- (void) setPassword: (NSString*)password;


/*!
    @method     entryEnumeratorSortedBy:
    @abstract   Returns an enumerator that contains the feed's entries sorted by the specified sort descriptors.
    @discussion This method allows the entries to be loaded into memory one at a time, which
                can make it more efficient than the regular entries method. (However, you
                also need to wrap your enumeration loop in an autorelease pool
                and drain the pool at the end of the loop. Otherwise, prior entries will not 
                be deallocated until sometime after the loop finishes.)

                If <code>sortDescriptors</code> is empty or <code>nil</code>, the entries will not be sorted.
    @param      sortDescriptors	An array containing a sort descriptor for sorting the feed's entries.
    @result     An enumerator containing each of the feed's entries, sorted by the given
                sort descriptors.
*/
- (NSEnumerator*) entryEnumeratorSortedBy: (NSArray*)sortDescriptors;



/*!
    @property   unreadCount
    @abstract   The number of unread entries in the feed. (read-only)
    @discussion To track this value (by displaying it in your UI, for example),
                call this method when this <code>PSFeed</code> sends a <code>PSFeedEntriesChangedNotification</code>.
*/
@property (readonly) int unreadCount;


/*!
    @method     entries
    @abstract   An array of entry objects that contains the entries of a feed. (read-only)
    @discussion Corresponds to <entry> in Atom and <item> in RSS.
*/
@property (readonly) NSArray * entries;


/*!
    @method     XMLRepresentation
    @abstract   Returns the raw feed, including entries.
    @discussion Equivalent to calling <code>XMLRepresentationWithEntries:YES</code>.
    @result     The raw feed including entry and article elements.
*/
- (NSXMLElement *) XMLRepresentation;


/*!
    @method     XMLRepresentationWithEntries:
    @abstract   Returns the raw feed, with or without the entry and article elements.
    @discussion The raw feed data is parsed into a DOM tree, and the root element (<feed> or <rss>)
                is returned. No other processing is performed.
                Setting the <code>withEntries</code> flag to <code>NO</code> causes the <entry> or <article> elements to be
                stripped out; other nonstandard child elements are kept, however.
    @param      withEntries  A boolean value to determine whether or not the resulting <code>NSXMLElement<code> object will
                contain the entry and article elements.
    @result     The raw feed.
*/
- (NSXMLElement *) XMLRepresentationWithEntries:(BOOL)withEntries;


/*!
    @method     extensionXMLElementsUsingNamespace:
    @abstract   Returns an array of the extension XML elements outside of the RSS and/or Atom standards.
    @discussion Pass <code>nil</code> for <code>nameSpace</code> to get all extension XML elements, 
                regardless of their namespace. This works at the feed-level only. You can use
                the same method on an entry to retrieve extension XML elements at that level.
                When using the <code>nameSpace</code> parameter, make sure to pass a full namespace, such as
                <code>http://www.itunes.com/dtds/podcast-1.0.dtd</code>, instead of the namespace prefix.
    @param      nameSpace  The namespace of the extension XML elements; pass <code>nil</code> to receive
                all extension XML elements.
    @result     An array of <code>NSXMLElement</code> objects containing the appropriate extension XML elements.
*/
- (NSArray *) extensionXMLElementsUsingNamespace:(NSString *)nameSpace;


@end


/* Notification posted by a PSFeed when it begins or ends refreshing.
   (Call -isRefreshing to get its new state.) */
extern NSString* const PSFeedRefreshingNotification NS_DEPRECATED(10_5, 10_9, NA, NA);


/* Notification posted by a PSFeed when entries are changed.
   The keys below appear in the userInfo dictionary, and their values are NSArrays of PSEntries,
   except for PSFeedRemovedEntriesKey whose value is an NSArray of entry identifier strings. */
extern NSString* const PSFeedEntriesChangedNotification NS_DEPRECATED(10_5, 10_9, NA, NA);

extern NSString* const PSFeedAddedEntriesKey NS_DEPRECATED(10_5, 10_9, NA, NA);
extern NSString* const PSFeedRemovedEntriesKey NS_DEPRECATED(10_5, 10_9, NA, NA);
extern NSString* const PSFeedUpdatedEntriesKey NS_DEPRECATED(10_5, 10_9, NA, NA);
extern NSString* const PSFeedDidChangeEntryFlagsKey NS_DEPRECATED(10_5, 10_9, NA, NA);

#endif
