//
//  VNRequest.h
//  Vision
//
//  Copyright © 2017 Apple Inc. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <Metal/Metal.h>
#import <Vision/VNObservation.h>


NS_ASSUME_NONNULL_BEGIN



/*!
 @header VNRequest The VNRequest class is the object a client uses to describe any computer vision task to be performed by the Vision framework. They are submitted to a request handler in order to perform the task, and the resulting VNObservations are placed in the results property of the VNRequest.
 */
@class VNRequest;

/*!
 @abstract A block that is executed at the completion of a request.
 @param request The VNRequest that has been completed. The results of the request if no error was encountered are populated in the results array of the request.
 @param	error The error that caused the request to fail, or nil if completed successfully.
 */
typedef void (^VNRequestCompletionHandler)(VNRequest *request, NSError * _Nullable error);


/*!
@class VNRequest
@abstract VNRequest objects describe the operation to be performed as well as act as the recipient of the operation's resultant observations.
 
@discussion VNRequest objects are instantiated in a pre-configured nominal state. Prior to sending a VNRequest to a request handler to perform a desired operation, the default configuration can be changed by modifying the values of VNRequest properties. The VNRequest class itself acts as a base class and is not meant to be directly instantiated.
*/
API_AVAILABLE(macos(10.13), ios(11.0), tvos(11.0))
@interface VNRequest : NSObject <NSCopying>


/*!
 @abstract Creates a new VNRequest with no completion handler.
*/
- (instancetype) init;


/*!
 @abstract Creates a new VNRequest with an optional completion handler.

 @param completionHandler	The block to be invoked after the request has completed its processing. The completion handler gets executed on the same dispatch queue as the request being executed.
*/
- (instancetype) initWithCompletionHandler:(nullable VNRequestCompletionHandler)completionHandler NS_DESIGNATED_INITIALIZER;


/*!
 @abstract A hint used to minimize the resource burden of the request. Memory footprint, processing footprint and/or CPU/GPU contention will be reduced (depending on the request), at the potential cost of longer execution time. This can help, for example, with ensuring UI updates and rendering are not getting blocked by Vision processing.
*/
@property (readwrite, nonatomic, assign) BOOL preferBackgroundProcessing;


/*!
 @abstract This property, if set to YES, signifies that the request should be performed exclusively on the CPU and not on the GPU. The default value is NO, which signifies that the request is free to leverage the GPU to accelerate any work the request may require.
 */
@property (readwrite, nonatomic) BOOL usesCPUOnly;

/*!
 @property results
 @abstract The collection of VNObservations generated by the processing of the request.
 @discussion The only valid time to access this property is after the request has been processed by a request handler.  If the request failed, this property will be nil; otherwise, it will be an array of zero or more VNObservation subclasses specific to the VNRequest subclass.
 */
@property (readonly, nonatomic, copy, nullable) NSArray *results;


/*!
 @property completionHandler
 @abstract The completion handler block that will be invoked after the request has completed processing.
 */
@property (readonly, nonatomic, copy, nullable) VNRequestCompletionHandler completionHandler;


@end




/*!
	@brief A request that will process the contents of a reference image.
*/
API_AVAILABLE(macos(10.13), ios(11.0), tvos(11.0))
@interface VNImageBasedRequest : VNRequest

/*!
	@brief The region of the image in which the request will be performed.  The rectangle is normalized to the dimensions of the image being processed and has its origin specified relative to the image's lower-left corner.
	
	@discussion The default value for this property is { { 0, 0 }, { 1, 1 } }.  Setting this property to a rectangle that is outside of the normalized coordinate space will be accepted but result in the request failing to be performed.
*/
@property (readwrite, nonatomic, assign) CGRect regionOfInterest;

@end




NS_ASSUME_NONNULL_END


